#***********************************************************************************************************************
# DMTF - Distributed Management Task Force, Inc. - http://www.dmtf.org
# See dsp2023_readme.txt for copyright information.
#
# embed-file.awk - part of the DMTF DSP2023 zip archive.
#
# awk script to process EMBED-FILE directives in HTML files generated by the DSP8029 XSLT stylesheet.
#
# Required tools:
#   cat (cat utility, from CygWin)
#   base64 (base64 encoder/decoder utility, from CygWin)
#
# Syntax of EMBED-FILE directives:
#
#   #EMBED-FILE filename 
#     -> resolved by replacing the line with the content of the file "filename".
#
#   #EMBED-FILE-AS-BASE64 filename 
#     -> resolved by replacing the line with the base64-encoded content of the file "filename".
#
#   #EMBED-CSS-FILE filename 
#     -> resolved by replacing the line with the content of the file "filename", stripping any C-language comments
#       (that is, text between "/*" and "*/")
#
#   These directives must appear in a line without anything else in that line.
#
#***********************************************************************************************************************

BEGIN {
    # make sure the embed-css-file_awk parameter is specified (in the command line using -v)
    if (!embed_css_file_awk) {
        system("echo 'embed-file.awk: Error: embed_css_file_awk parameter is not specified to awk'>&2")
	    exit(1)
    }
}

# Process a line with "#EMBED-FILE(file)":
/^.*#EMBED-FILE\(.*\).*$/ {

    match( $0, /(.*)#EMBED-FILE\((.*)\)(.*)/, result)
    line_before = result[1]
    file = result[2]   
    line_after = result[3]
	 
    # print "Debug: Found EMBED-FILE(" file ")"

	resolved_file = uri_resolve(file)
	
    printf "%s", line_before

    # Output the content of the file.
    cmd = "cat '"resolved_file"'"
    system("echo 'embed-file.awk: "cmd"'>&2")
	rc = system(cmd)
    if (rc != 0) {
        system("echo 'embed-file.awk: Error: rc="rc"'>&2")
	    exit(1)
	}    

    printf "%s\n", line_after

    # Drop the matching line.
    next
}

# Process a line with "#EMBED-FILE-AS-BASE64(file)":
/^.*#EMBED-FILE-AS-BASE64\(.*\).*$/ {

    match( $0, /(.*)#EMBED-FILE-AS-BASE64\((.*)\)(.*)/, result)
    line_before = result[1]
    file = result[2]   
    line_after = result[3]
	 
    # print "Debug: Found EMBED-FILE-AS-BASE64(" file ")"

	resolved_file = uri_resolve(file)

    printf "%s", line_before
	
    # Output the base64-encoded content of the file.
    cmd = "base64 <'"resolved_file"'"
    system("echo 'embed-file.awk: "cmd"'>&2")
	rc = system(cmd)
    if (rc != 0) {
        system("echo 'embed-file.awk: Error: rc="rc"'>&2")
	    exit(1)
	}    

    printf "%s\n", line_after

    # Drop the matching line.
    next
}

# Process a line with "#EMBED-CSS-FILE(file)":
/^.*#EMBED-CSS-FILE\(.*\).*$/ {

    match( $0, /(.*)#EMBED-CSS-FILE\((.*)\)(.*)/, result)
    line_before = result[1]
    file = result[2]   
    line_after = result[3]
	 
    # print "Debug: Found EMBED-CSS-FILE(" file ")"

	resolved_file = uri_resolve(file)
	
    printf "%s", line_before

    # Output the base64-encoded content of the file.
    cmd = "gawk -f "embed_css_file_awk" <'"resolved_file"'"
    system("echo 'embed-file.awk: "cmd"'>&2")
	rc = system(cmd)
    if (rc != 0) {
        system("echo 'embed-file.awk: Error: rc="rc"'>&2")
	    exit(1)
	}    

    printf "%s\n", line_after

    # Drop the matching line.
    next
}

# Process any other lines:
{
    # Output the line unchanged.
    print $0
}

function uri_resolve (uri)
{
    tmp_file = "uri.tmp"
    
	cmd = "uriresolve.bat -u "uri" uri >"tmp_file
    # system("echo 'embed-file.awk: Debug: cmd="cmd"'>&2")
	rc = system(cmd)
    if (rc != 0) {
        system("echo 'embed-file.awk: Error: uriresolve.bat failed with rc="rc"'>&2")
	    exit(1)
	}    

	# In the success case (rc=0), the uriresolve batch script outputs multiple lines on stdout.
	# The second word in the line starting with the string "Result:" is the resolved URI, or
	# the string "null" if the input URI could not be resolved.

	resolved_uri = ""
    while (getline output_line < tmp_file == 1)
    {
        if (index(output_line, "Result:") == 1) {
            split(output_line, output_word, " ")
		    resolved_uri = output_word[2]
		}    
    }
    close(tmp_file)

    if (resolved_uri == "") {
        system("echo 'embed-file.awk: Internal Error: Cannot parse output of uriresolve.bat (see "tmp_file")'>&2")
	    exit(1)
	}    
	else {
        system("rm "tmp_file)
	}

    if (resolved_uri == "null") {
		# If the URI could not be resolved, use the unresolved input URI.
        # system("echo 'embed-file.awk: Debug: Did not find URI resolution for "uri", using it unchanged'>&2")
        resolved_uri = uri
	}    
	else {
        # system("echo 'embed-file.awk: Debug: Resolved uri "uri" to "resolved_uri"'>&2")
	}

    if (index(resolved_uri, "file:/") == 1) {
        match(resolved_uri, "file:/+(.*)", file_array)
        resolved_file = file_array[1]
        # system("echo 'embed-file.awk: Debug: file scheme in URI, resolved_file="resolved_file"'>&2")
	}    
    else if (index(resolved_uri, "http:/") == 1) {
        system("echo 'embed-file.awk: Error: Cannot use http-based URI for EMBED-FILE directives: "resolved_uri"'>&2")
	    exit(1)
	}    
    else {
        # We assume the URI is a relative path (probably the original uri)
        resolved_file = resolved_uri
	}    

    # Translate file URIs back to Windows path names
    # gsub("/", "\\\\", resolved_file)
    gsub("%20", " ", resolved_file)
    if (index(resolved_uri, "%") > 0) {
        system("echo 'embed-file.awk: Internal Error: Missing support for percent-unescaping in URI: "resolved_uri"'>&2")
	    exit(1)
	}    

    # system("echo 'embed-file.awk: Debug: returning resolved_file="resolved_file"'>&2")

	return resolved_file
}